/* PDB2PIM converts the HP 200LX PDB database (phone book) in conjunction
 * with GDBIO so that the output can be used as PIM/PE contacts database
 *
 * (c) 2001 by Daniel Hertrich http://www.daniel-hertrich.de/pdb2x
 * 
 * needed: GDBIO (http://www.palmtop.net/super.html)
 *         PIM/PE by Andreas Garzotto (ftp://ftp.dasoft.com).
 *
 * usage:  prepare phone book file with
 *
 *           GDBIO /X /N C:\_DAT\PHONE.PDB > PHONEWN.CDF
 *
 *         (the WN in phoneWN.cdf refers to "with notes".
 *         Be sure to create a CDF file containing all notes
 *         using the GDBIO parameters /X /N, otherwise PDB2PIM
 *         cannot work properly!)
 *
 *         Then do a
 *
 *           GDBIO C:\_DAT\PHONE.PDB /S | more
 *       
 *         This will show you the structure of your phone database.
 *         Determine the field numbers corresponding to the
 *         fields you want to see in your PIM file.
 *         You can use up to 20 fields
 *
 *         So, call for example
 *
 *         PDB2PIM PHONEWN.CDF +p 1 5 3 > contacts.pim
 *
 *         This will creat a contacts.pim file which contains records
 *         of the following style:
 *
 *         +p [FIELD1]
 *            [FIELD5]
 *            [FIELD3]
 *            | [NOTE LINES]
 *            | [NOTE LINES]
 *
 *         In the output file empty fields of the input file will not
 *         create an empty line. They are simply omitted.
 *
 *         The output will not be sorted alphabetically!
 *         In order to create a sorted output, the CDF input file
 *         would have to be sorted. This can be done using SED and BIGSORT:
 *
 *            SED -e :a -e "$!N;s/\nN/@~@N/;ta" -e "P;D" PHONEWN.CDF > PHONEWNS.TMP
 *            BIGSORT /i < PHONEWNS.TMP > PHONEWN1.TMP
 *            SED "s/@~@N/\nN/g" PHONEWN1.TMP > PHONEWNS.CDF
 *            DEL *.TMP
 *
 *         --> PHONEWNS.CDF contains the CDF with notes in sorted order.
 *             (What this complicated set of commands does is:
 *              - Combine all lines of each record in PHONEN.CDF to one single 
 *                line by replacing the newlines within one record 
 *                with the string "@~@"
 *              - Sort the lines using bigsort
 *              - Replacing the "@~@" with newlines again to create the
 *                original record structure
 *             This method assumes you do not have the string "@~@" somewhere in 
 *             your phone book!)
 *
 */
 
#define VERSION "2.2"
#define YEAR "2003"
#define TOKLENGTH 80    // maximal length for a token, such as email, phone no, name of phone.cdf
#define LINELENGTH 1024  // maximal length of a line in cdf file

#include <stdlib.h>
#include <stdio.h>
#include <string.h>


FILE *fpi;
int i,j;
int linecounter=0;
char testchar;
int field[20];
int maxfield;
char line[LINELENGTH];
char line2[LINELENGTH-3];
char delimiter[3];
char *dup;
char pimline[TOKLENGTH];

void hello(char *filename);
void usage (void);
char *extr_cdf_col(char *cdfline, int c);
void process_file(int field[20], int maxfield);

/* MAIN ============================================== */

int main(int argc, char *argv[])
{
  if (argc < 4) usage();

  strcpy(delimiter,argv[2]);

  for (i=2;i<argc;i++)
  {
    field[i-2] = atoi(argv[i]);
  }
  maxfield = i-3;

  hello(argv[1]);
  
  if ((fpi=fopen(argv[1],"r"))!=NULL)
  {
    printf("+. Contacts.pim\n");
    process_file(field, maxfield);
    fclose(fpi);
  }

  else 
  {
    fprintf(stderr,"\n\aError opening input file \"%s\"!\nAbnormal program termination.\n\n",argv[1]);
    return(1);
  }
  return(0);
}


/* HELLO ============================================= */

void hello(char *filename)
{
  if (filename == NULL)
  {
    filename="NO INPUT FILE NAME GIVEN!";
  }
  fprintf(stderr,"\n*** This is PDB2PIM Version %s, (c) %s Daniel Hertrich\n",VERSION,YEAR);
  fprintf(stderr,"*** Input CDF file: %s\n",filename);
  fprintf(stderr,"*** Field numbers used for output: ");
  for (i=0;i<=maxfield;i++)
  {
    fprintf(stderr,"%i ",field[i]);
  }
  fprintf(stderr,"\n\n");
  
}


/* USAGE =========================================== */

void usage (void)
{
  fprintf(stderr,"\n*** This is PDB2PIM, Version %s, (c) %s Daniel Hertrich",VERSION,YEAR);
  fprintf(stderr,"\n*** Call: PDB2PIM <CDFFILE> <delimiter> <f1 [f2 [f3 [..] ] ] > > <output.pim> ");
  fprintf(stderr,"\n*** 1 to 20 field numbers can be given. Delimiter is usually \"+p\"");
  fprintf(stderr,"\n*** Details see pdb2pim.c source file.\n\n");
  exit(1);
}


/* EXTR_CDF_COL ===================================== */
/* give parameters:
      1. copy of a line of a cdf file (made by gdbio)
      2. number of column you want to extract
   get back:
      pointer to string containing string of given column
*/

char *extr_cdf_col(char *cdfline, int c)
{
  static char token[TOKLENGTH];
  char *loc;
  char *temptok = malloc(LINELENGTH*sizeof(char));
  int i = 0;

  temptok[0]=0;
    
  for (i=0; i<c; i++)
  {
    if (i>0) cdfline=loc+3;
    loc=strstr(cdfline,"\",\"");
    if (loc != NULL)
    { 
      strncpy(temptok,cdfline,loc-cdfline);
      temptok[loc-cdfline]='\0';
    }
    else
    {
      fprintf(stderr,"ERROR! Contact Author! loc is NULL!, c is %i\nCDFLINE is: %s\nLine number: %i\n\n\a\a",c,cdfline,linecounter);
      exit(1);
    }
  }
  strcpy(token,temptok);
  free(temptok);
  return(token);
}


/* PROCESS_FILE ========================================= */

void process_file(int fields[20], int maxfield)
{
  while((testchar=fgetc(fpi))!= EOF)
  {
    linecounter++;
    fseek(fpi,-1,SEEK_CUR);                     // go one character back
    fgets(line, LINELENGTH, fpi);

    if (line[0] == 'D')
    {
      for (i=0;i<LINELENGTH-3;i++)
      {
        line2[i] = line[i+3];
      }
      printf("\n%s ",delimiter);

      for (i=0;i<=maxfield;i++)
      {
        dup=strdup(line2);
        strcpy(pimline,extr_cdf_col(dup,fields[i]));
        if (!strlen(pimline)==0)
        {
          if (i==1) printf("%s\n",pimline);
          else  printf("   %s\n",pimline);
        }
        free(dup);
     }
    }
    
    else
		{
      for (i=0;i<LINELENGTH-2;i++)
      {
        line2[i] = line[i+2];
      }
      printf("   | %s",line2);
    }
  }
}



