#include "sjog.h"
#include "sjog_brightness.h"
#include "sjog_brightness_callbacks.h"

#define MUL 2.55

/* -------------------------------------------------------------------------- */

GtkWidget *brightnessWindow; /* brightness dialog box */
GtkWidget *brightnessProgressBar;	/* widget to show brightness level */
guint brightnessTimer;       /* window hide timer  */
guint currentBrightness = 50;	/* startup brightness */

/* -------------------------------------------------------------------------- */

/* Called by sjog_check_jog_position when we are in SJOG_MODE_BRIGHTNESS.
 * Checks the jog and increase/decrease the brightness level according to
 * its position.
 */
void
sjog_brightness_check_jog_position(int jog)
{

#ifdef DEBUG
   fprintf(stdout,
           "sjog_brightness_check_jog_position: currentBrightness=%d\n",
           currentBrightness);
#endif

   switch (jog)
   {

        /* decrease brightness */
     case JOG_BOTTOM:
        sjog_reset_brightness_timer();
        currentBrightness -= 10;
        if ((int) currentBrightness < 0)
           currentBrightness = 0;
        sjog_set_brightness(currentBrightness);
        break;

        /* hide brightness window */
     case JOG_PUSHED:
        gtk_timeout_remove(brightnessTimer);
        sjog_brightness_hide();
        break;

        /* increase brightness */
     case JOG_TOP:
        sjog_reset_brightness_timer();
        currentBrightness += 10;
        if (currentBrightness > 100)
           currentBrightness = 100;
        sjog_set_brightness(currentBrightness);
        break;

     default:

   }

}

/* -------------------------------------------------------------------------- */

/*
 *
 */
int
sjog_get_brightness()
{
   FILE *pin;
   int brightness = 50;
   char buf[10];

   if ((pin = popen("spicctrl -B", "r")))
   {
      if (fgets(buf, sizeof(buf) - 1, pin) != NULL)
      {
         brightness = (int) (atoi(buf) / MUL);
      }
      pclose(pin);
   }
#ifdef DEBUG
   fprintf(stdout, "sjog_get_brightness: val=%d\n", brightness);
#endif

   if (reverseBrightness)
      brightness = 100 - brightness;

   return brightness;
}

/* Takes a new brightness value and sets the screen brightness to this
 * value. This function uses the setbrightness external program which
 * must be setuid root to work.
 */
void
sjog_set_brightness(int brightness)
{

   gchar *cmd = NULL;

   /* setbrightness takes an argument value between 0 and 255
    * 0 means dark , 255 means light
    * or vice-versa depending on the vaio model
    */
   if (reverseBrightness)
      brightness = MUL * 100 - (int) brightness *MUL;

   else
      brightness = (int) brightness *MUL;


   /* ensure in range */
   brightness = (brightness < 0 ? 0 : (brightness > 255 ? 255 : brightness));

   /* setup the command line, setbrightness must be in PATH */
   cmd = g_strdup_printf("spicctrl -b %d", brightness);

#ifdef DEBUG
   fprintf(stdout, "sjog_set_brightness: cmd=%s\n", cmd);
#endif

   /* execute setbrightness */
   system(cmd);

   g_free(cmd);

   /* finally sets the progress bar to the new brightness value */
   gtk_progress_set_value(GTK_PROGRESS(brightnessProgressBar),
                          currentBrightness);

}

/* -------------------------------------------------------------------------- */

/* Shows the screen brightness level setting dialog.
 */
void
sjog_brightness_show()
{

   /* first launch needs gui initialization */
   if (brightnessWindow == NULL)
      init_brightness_gui();

   /* forces brightness to default value */
   sjog_set_brightness(currentBrightness);

   gtk_widget_show_all(brightnessWindow);

   /* initializes the hide timer */
   sjog_reset_brightness_timer();
}

/* -------------------------------------------------------------------------- */

/* Shows the screen brightness level setting dialog.
 */
void
sjog_brightness_hide()
{
   gtk_timeout_remove(brightnessTimer);
   gtk_widget_hide_all(brightnessWindow);
   sjog_set_mode(SJOG_MODE_SCROLL);
}

/* -------------------------------------------------------------------------- */

/* Remove existing brightness timer if exists and set a new one.
 */
void
sjog_reset_brightness_timer()
{
   if (brightnessTimer != 0)
      gtk_timeout_remove(brightnessTimer);
   brightnessTimer =
      gtk_timeout_add(SJOG_TIMEOUT,
                      (GtkFunction) sjog_brightness_timer_callback, NULL);
}

/* -------------------------------------------------------------------------- */

/* Builds the widgets of the brightness dialog window
 * Widgets hierarchy:
 *      brightnessWindow -> vbox -> brightness_pixmap
 *                               -> brightnessProgressBar
 */
void
init_brightness_gui()
{

   GtkAdjustment *adjust;
   GtkWidget *vbox;
   GdkPixmap *gd_pixmap;
   GtkWidget *brightness_pixmap;

   /* vertical box */
   vbox = gtk_vbox_new(FALSE, 10);

   /* brightness window */
   brightnessWindow = gtk_window_new(GTK_WINDOW_DIALOG);
   gtk_container_set_border_width(GTK_CONTAINER(brightnessWindow), 10);

   sjog_window_set_position(GTK_WINDOW(brightnessWindow));

   gtk_window_set_title(GTK_WINDOW(brightnessWindow), "S-Jog: Brightness");
   gtk_signal_connect(GTK_OBJECT(brightnessWindow), "delete-event",
                      GTK_SIGNAL_FUNC
                      (sjog_brightness_signal_delete_event_callback), NULL);
   gtk_container_add(GTK_CONTAINER(brightnessWindow), vbox);

   /* brightness image */
   gtk_widget_realize(brightnessWindow);
   gd_pixmap =
      gdk_pixmap_create_from_xpm(gtk_widget_get_parent_window(vbox), NULL,
                                 NULL,
                                 strdup(PKGDATADIR
                                        "/pixmaps/sjog-brightness.xpm"));

   brightness_pixmap = gtk_pixmap_new(gd_pixmap, NULL);
   gtk_container_add(GTK_CONTAINER(vbox), brightness_pixmap);

   /* brightness progress bar */
   adjust =
      (GtkAdjustment *) gtk_adjustment_new(DEFAULT_VALUE, MIN_VALUE,
                                           MAX_VALUE, STEP_INC, PAGE_INC,
                                           PAGE_SIZE);
   brightnessProgressBar = gtk_progress_bar_new_with_adjustment(adjust);
   gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR(brightnessProgressBar),
                                    GTK_PROGRESS_LEFT_TO_RIGHT);
   gtk_progress_bar_set_bar_style(GTK_PROGRESS_BAR(brightnessProgressBar),
                                  GTK_PROGRESS_CONTINUOUS);
   gtk_progress_set_format_string(GTK_PROGRESS(brightnessProgressBar),
                                  "%v%%");
   gtk_progress_set_show_text(GTK_PROGRESS(brightnessProgressBar), TRUE);

   gtk_container_add(GTK_CONTAINER(vbox), brightnessProgressBar);

   currentBrightness = sjog_get_brightness();
}
